% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                        %
% Realizamos una exploracion previa de un problema de ODE con CC.        %
%                                                                        %
% Consideramos el caso de una catenaria de densidad de masa por          %
% unidad de longitud rho de dos postes separados una distancia L y       %
% con una diferencia de altura entre ellos igual a dh.                   %
%                                                                        %
% Para ello, integramos mediante una rutina de integracion de ODEs       %
% con CI la ecuacion de la catenaria para varios valores de la derivada  %
% en el punto x=0.                                                       %
%                                                                        %
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% Y ^
%   |
%   |
%   |
%   |
%   |
%   |
%   |
%   |(xa,ya)                                   (xb,yb)
%  -o--------------------------------------------o---------->
%    o  theta                                  o           X
%      o                                     o
%         o                               o
%             o                       o
%                  o             o
%                         o  
%
%   |------------------   L   -------------------|
%
%

clear; clc;  % limpieza del espacio de trabajo

% PARAMETROS FISICOS DE ENTRADA
%
% dh    diferencia de altura de los postes.
%
% L     separacion horizontal entre los puntos de los que pende 
%       la catenaria
%
% rho   densidad de masa por unidad de longitud de la cuerda multiplicada
%       por la aceleracion de la gravedad
%
% Th    componente horizontal de la tension en los puntos de suspension
%
% (xa, ya) (xb, yb) coordenadas de los puntos de suspension

dh = 0.0;
L = 10.0; 
rho = 20.0;
Th   = 50.0;

xa = 0.0;
xb = L;
ya = 0.0;
yb = dh;
       
% PARAMETROS PARA LA EXPLORACION NUMERICA
% Probamos de 5 en 5 grados (expresados en radianes).
% 
% theta   angulo entre la horizontal y la tangente a la catenaria en
%         en punto de suspension. Lo tomamos de 5 en 5 grados. 
%
% nx      numero de valores (x,y) en los que queremos la solucion

theta = -5.0:-5.0:-85.0;
nx = 50; 
                  

% EXPLORACION DEL METODO DE DISPARO
%
% Realizamos varias integraciones del problema de condiciones iniciales
% definido por la ecuacion de la catenaria con CI y(0) = 0, y'(0)=s.
%
% Para la ingracion del problema de CI utilizamos la rutina ode45 de 
% matlab. Esta rutina requiere como argumento una función que le 
% proporcione las derivadas de las variables. En el caso de la funcion
% eccatenaria, sobra el parametro C, de modo que definimos una funcion
% fcat auxiliar para eliminarlo.
%
% fcat: funcion anonima auxiliar para pasar la ecuacion diferencial de la 
%       catenaria (definida en su propio fichero) como argumento a ode45. 
%
% Los valores de las coordenadas horizontales en los que queremos solucion
% los elegimos equiespaciados.
%
% Guardamos en el array deltaobj la 'desviacion' del disparo, i.e. la 
% diferencia entre el resultado obtenido y el deseado.
%

fcat = @(x,y) eccatenaria(rho/Th, y);
x = linspace( xa, xb, nx );
deltaobj = zeros(1, length(theta));

figure(1); clf;

for ii = 1:length(theta)
    
    s = tan( theta(ii)*pi/180.0 );
    y0 = [ ya; s ];
    [xs, ys] = ode45( fcat, x, y0);
    
    % Dibujamos la solucion y(x).
    % Destacamos el valor y(x=L) con un marcador rojo.
    % Anotamos el valor del angulo al que corresponde.
    
    plot(xs, ys(:,1),'-',xs(nx),ys(nx,1),'or');
    leyenda = sprintf('%dº',theta(ii));
    text( xs(nx),ys(nx,1),leyenda);
    pause(0.25);
    hold on;
    deltaobj(ii) = ys(nx,1) - yb;
end

grid;
xlabel('x');
ylabel('y');

figure(2); clf;
plot(theta,deltaobj,'o-r');
grid;
xlabel('theta');
ylabel('deltaobj');
%